#!/bin/bash

# generate the file structure and the configuration file
# to host a new Virtual Host with Apache HTTPD

# this script has been created for an Ubuntu system
# (specific commands, files and folder names)

# Xavier Belanger
# May 14, 2025

# root-level privileges are required for this script
if [ $EUID -ne 0 ]
then
	/usr/bin/printf "You must use sudo to run this script\n"
	exit 1
fi

# the script requires a full hostname as an argument
if [ $# -lt 1 ]
then
	/usr/bin/printf "usage: $0 FQDN\n"
	exit 1
fi

# using the script argument as the full hostname, in lower case
fullHostname=$(echo $1 | /usr/bin/tr [:upper:] [:lower:])

# checking if a website doesn't already use that hostname
# Warning: this checks only for configuration files generated by this script
confFilesCount=$(/usr/bin/find /etc/apache2/sites-available/ -type f -name '*https.conf' | /usr/bin/wc -w)
if [ $confFilesCount -gt 0 ]
then
	existingSites=$(/bin/grep -io "ServerName $fullHostname" /etc/apache2/sites-available/*https.conf | /usr/bin/wc -l)

	if [ "$existingSites" -ne "0" ]
	then
		/usr/bin/printf "ERROR: this hostname is already in use\n"
		exit 1
	fi
fi

# using the hostname for files and directories
# note: subdomains are ignored ('something.else.example.com' will be shortened as 'something')
shortHostname=$(echo $fullHostname | /usr/bin/cut -d "." -f1)

webDirectory=/var/www/$shortHostname

# creating the website directory, if it doesn't already exist
if [ ! -r $webDirectory ]
then
	/bin/mkdir $webDirectory
fi

# generating a placeholder HTML file, if it doesn't already exist
indexFile=$webDirectory/index.html

if [ ! -r $indexFile ]
then
	(/bin/cat << EOF
<!DOCTYPE html>
<html lang="en">
<head><title>$fullHostname</title></head>
<body><h1>Welcome to the $fullHostname website<h1></body>
</html>
EOF
	) > $indexFile
	/usr/bin/printf "$indexFile has been generated\n"
fi

# generating a 401 HTML file, if it doesn't already exist
page401File=$webDirectory/401.html

if [ ! -r $page401File ]
then
        (/bin/cat << EOF
<!DOCTYPE html>
<html lang="en">
<head><title>$fullHostname :: 401 Unauthorized</title></head>
<body><p>You have not provided the proper credentials to access to this document.</p></body>
</html>
EOF
        ) > $page401File
	/usr/bin/printf "$page401File has been generated\n"
fi

# generating a 403 HTML file, if it doesn't already exist
page403File=$webDirectory/403.html

if [ ! -r $page403File ]
then
        (/bin/cat << EOF
<!DOCTYPE html>
<html lang="en">
<head><title>$fullHostname :: 403 Forbidden</title></head>
<body><p>You can't have access to the requested document.</p></body>
</html>
EOF
        ) > $page403File
	/usr/bin/printf "$page403File has been generated\n"
fi

# generating a 404 HTML file, if it doesn't already exist
page404File=$webDirectory/404.html

if [ ! -r $page404File ]
then
        (/bin/cat << EOF
<!DOCTYPE html>
<html lang="en">
<head><title>$fullHostname :: 404 Not Found</title></head>
<body><p>The requested document doesn't exist.</p></body>
</html>
EOF
        ) > $page404File
	/usr/bin/printf "$page404File has been generated\n"
fi

# fixing ownership and permissions
/bin/chown -R root:www-data $webDirectory
/bin/chmod -R g+w $webDirectory

# using a minimal website configuration file template
configTemplate=$(/bin/mktemp /tmp/configTemplate-XXXX)

(/bin/cat << EOF
# Generated by the mkwebsite-https.sh script
# on $(/bin/date +"%A, %B %e, %Y")

<VirtualHost *:443>
	ServerName example.com
	ServerAlias www.example.com
	ServerAdmin webmaster@example.com
	DocumentRoot /var/www

	ErrorDocument 401 /401.html
	ErrorDocument 403 /403.html
	ErrorDocument 404 /404.html

        ErrorLog \${APACHE_LOG_DIR}/error.log
        CustomLog \${APACHE_LOG_DIR}/access.log combined

	SSLEngine on
	SSLCertificateFile /etc/apache2/sites-certificates/example.com-cert.pem
	SSLCertificateKeyFile /etc/apache2/sites-certificates/example.com-privkey.pem
	SSLCertificateChainFile /etc/apache2/sites-certificates/example.com-chain.pem

	<FilesMatch "\.php$">
		SSLOptions +StdEnvVars
	</FilesMatch>

	RewriteEngine On
	RewriteCond %{REQUEST_METHOD} ^OPTIONS
	RewriteRule .* - [F]
</VirtualHost>

# EoF
EOF
) > $configTemplate

# the website configuration file is placed in the Apache configuration directory
siteConfigFile=/etc/apache2/sites-available/$shortHostname-https.conf
/bin/mv $configTemplate $siteConfigFile
/bin/chmod 644 $siteConfigFile

# editing the site configuration file
# defining the ServerName and ServerAlias
/bin/sed -i "s/ServerName example.com/ServerName $fullHostname/" $siteConfigFile
/bin/sed -i "s/ServerAlias www.example.com/ServerAlias www.$fullHostname/" $siteConfigFile

# updating the webmaster email's address
/bin/sed -i "s/ServerAdmin webmaster@example.com/ServerAdmin webmaster@$fullHostname/" $siteConfigFile

# updating the DocumentRoot
/bin/sed -i "s/DocumentRoot \/var\/www/DocumentRoot \/var\/www\/$shortHostname/" $siteConfigFile

# updating the logfile names
/bin/sed -i "s/error.log/$shortHostname-https-error.log/" $siteConfigFile
/bin/sed -i "s/access.log/$shortHostname-https-access.log/" $siteConfigFile

# updating the certificate file names
/bin/sed -i "s/example.com-cert.pem/www.$fullHostname-cert.pem/" $siteConfigFile
/bin/sed -i "s/example.com-privkey.pem/www.$fullHostname-privkey.pem/" $siteConfigFile
/bin/sed -i "s/example.com-chain.pem/www.$fullHostname-chain.pem/" $siteConfigFile

# checking the resulting files
/usr/bin/printf "\nConfiguration file: \n"
if [ -r $siteConfigFile ]
then
	/bin/cat $siteConfigFile
else
	/usr/bin/printf "Error accessing the configuration file\n"
fi

/usr/bin/printf "\nPlaceholder web files: \n"
for webfile in "$indexFile" "$page401File" "$page403File" "$page404File"
do
	if [ -r $webfile ]
	then
		/usr/bin/printf "$webfile is in place\n"
	else
		/usr/bin/printf "Error accessing the $webfile file\n"
	fi
done

# providing instructions for activating the web site
/usr/bin/printf "\nIn order to activate the web site, please run the following command:\n"
/usr/bin/printf "  sudo a2ensite $(/usr/bin/basename $siteConfigFile .conf)\n"

# EoF
